const { SERVER_API } = window.CONFIG;

// Biến toàn cục
let allRules = [];
let idRewardRule = null;
let mode = "add"; // mode cho Rule Modal
let currentDeleteId = null; // ID khách hàng cần xóa

// Helper: Định dạng tiền tệ
function formatCurrency(amount) {
  if (!amount && amount !== 0) return "0 ₫";
  return new Intl.NumberFormat("vi-VN", {
    style: "currency",
    currency: "VND",
  }).format(amount);
}

// Helper: Đổi số thành chữ
function numberToWords(num) {
  if (isNaN(num)) return "";
  return new Intl.NumberFormat("vi-VN").format(num) + " đồng";
}

// Helper: Format ngày tháng
function formatDate(dateString) {
  if (!dateString) return "";
  const date = new Date(dateString);
  return new Intl.DateTimeFormat("vi-VN", {
    day: "2-digit",
    month: "2-digit",
    year: "numeric",
  }).format(date);
}

(function showToday() {
  const today = new Date();
  const day = String(today.getDate()).padStart(2, "0");
  const month = String(today.getMonth() + 1).padStart(2, "0");
  const year = today.getFullYear();
  const el = document.getElementById("today");
  if (el) el.textContent = `${day}-${month}-${year}`;
})();

window.openDeleteModal = function (id) {
  currentDeleteId = id;
  const modal = new bootstrap.Modal(
    document.getElementById("deleteCustomerModal")
  );
  modal.show();
};

window.renderCarousel = function (data) {
  console.log("Rendering Carousel...", data);
  const carouselInner = document.querySelector(
    "#carouselExample .carousel-inner"
  );

  if (!carouselInner) return;

  carouselInner.innerHTML = ""; // Clear cũ

  if (!data || data.length === 0) {
    carouselInner.innerHTML = `
            <div class="text-center py-5">
                <i class="bi bi-card-image text-secondary fs-1 mb-2 opacity-50"></i>
                <p class="text-muted">Chưa có hình ảnh vinh danh</p>
            </div>`;
    // Ẩn nút điều hướng nếu không có dữ liệu
    const prevBtn = document.querySelector(".carousel-control-prev");
    const nextBtn = document.querySelector(".carousel-control-next");
    if (prevBtn) prevBtn.classList.add("d-none");
    if (nextBtn) nextBtn.classList.add("d-none");
    return;
  }

  // Hiện nút điều hướng
  document.querySelector(".carousel-control-prev")?.classList.remove("d-none");
  document.querySelector(".carousel-control-next")?.classList.remove("d-none");

  data.forEach((item, index) => {
    const isActive = index === 0 ? "active" : "";
    
    // Xử lý ảnh: Check link http hoặc đường dẫn tương đối
    let avatarUrl = "../../images/heovang.png"; // Ảnh mặc định
    if (item.image_url) {
        avatarUrl = item.image_url.startsWith("http") 
            ? item.image_url 
            : `https://bigapptech.vn${item.image_url}`;
    }

const itemHtml = `
    <div class="carousel-item ${isActive}">
        <div class="winner-card shadow-sm border rounded bg-white" style="max-width: 450px; margin:auto; overflow: hidden;">
            <div class="position-relative">
                <img src="${avatarUrl}" class="winner-banner-img w-100" 
                     style="height: 190px; object-fit: cover;" alt="${item.name_customer}">
                
                <div class="position-absolute top-0 end-0 m-2">
                    <span class="badge bg-dark bg-opacity-75 text-white py-2 px-3 small">
                        <i class="bi bi-calendar3 me-2"></i>${formatDate(item.received_at)}
                    </span>
                </div>
            </div>

            <div class="winner-info p-3 text-center">
                <h5 class="fw-bold text-dark mb-1 text-truncate">${item.name_customer}</h5>
                <p class="text-muted small mb-2 fst-italic">Đã xuất sắc nhận thưởng</p>
                
                <div class="d-flex justify-content-center align-items-center mb-3">
                    <span class="badge bg-warning text-dark fs-6 px-4 py-2 rounded-pill shadow-sm border border-warning">
                        ${formatCurrency(item.amount_received)}
                    </span>
                </div>
                
                <div class="d-flex justify-content-center gap-2 mt-2">
                     <button class="btn btn-sm btn-light border btn-edit-customer" 
                        data-id="${item.id}"
                        data-club_id="${item.club_id}"
                        data-amount_received="${item.amount_received}"
                        data-name_customer="${item.name_customer}"
                        data-received_at="${item.received_at}"
                        data-image_url="${avatarUrl}">
                        <i class="bi bi-pencil-square text-primary"></i> Sửa
                    </button>
                    <button class="btn btn-sm btn-light border" 
                            onclick="openDeleteModal('${item.id}')">
                        <i class="bi bi-trash3 text-danger"></i> Xóa
                    </button>
                </div>
            </div>
        </div>
    </div>
`;
    carouselInner.insertAdjacentHTML("beforeend", itemHtml);
  });


  const carouselEl = document.getElementById("carouselExample");
  const bsCarousel = bootstrap.Carousel.getInstance(carouselEl);
  if (bsCarousel) bsCarousel.dispose();
  new bootstrap.Carousel(carouselEl);

  bindEditButtons();
};


function renderRule(gameMode) {
  const rule = allRules.find((r) => r.game_mode == gameMode);
  
  if (rule) {
    idRewardRule = rule.id;

    const amountInput = document.getElementById("amount_received");
    if (amountInput) {

      amountInput.setAttribute("max", rule.total_amount);
    }

    const priceText = document.querySelector(".price-text");
    if (priceText) priceText.textContent = formatCurrency(rule.total_amount);
    
    const maxTotal = document.querySelector(".max_current_total_amount");
    if (maxTotal) maxTotal.textContent = formatCurrency(rule.total_amount);

    const ruleDetails = document.querySelector(".content-reward-rule");
    if (ruleDetails) ruleDetails.innerHTML = rule.rule_details || "Chưa có nội dung";

    const rewardContent = document.querySelector(".content-reward");
    if (rewardContent) rewardContent.innerHTML = rule.reward_content || "Chưa có nội dung";

    document.querySelector(".btn-add-reward-rule")?.classList.add("d-none");
    document.querySelector(".btn-edit-reward-rule")?.classList.remove("d-none");
    document.querySelector(".btn-delete-reward-rule")?.classList.remove("d-none");
  } else {
    document.querySelector(".price-text").textContent = formatCurrency(0);
    document.querySelector(".max_current_total_amount").textContent = formatCurrency(0);
    document.querySelector(".content-reward-rule").innerHTML = "";
    document.querySelector(".content-reward").innerHTML = "";

    document.querySelector(".btn-add-reward-rule")?.classList.remove("d-none");
    document.querySelector(".btn-edit-reward-rule")?.classList.add("d-none");
    document.querySelector(".btn-delete-reward-rule")?.classList.add("d-none");
  }
}

async function loadWinners(clubId, gameMode) {
  if (!clubId) return;
  const carouselInner = document.querySelector(
    "#carouselExample .carousel-inner"
  );
  if (carouselInner)
    carouselInner.innerHTML =
      '<div class="text-center py-5"><div class="spinner-border text-primary"></div></div>';

  try {
    const url = `${SERVER_API}/api/bigscore/customer/get-by-id-club-and-game-mode?club_id=${clubId}&game_mode=${gameMode}`;
    const res = await fetch(url, {
      method: "GET",
      headers: { Accept: "application/json", "X-CLIENT-REQUEST": "HERO" },
    });
    const data = await res.json();

    if (data.code === 200) {
      window.renderCarousel(data.data);
    } else {
      window.renderCarousel([]);
    }
  } catch (error) {
    console.error("Lỗi tải danh sách trúng thưởng:", error);
    window.renderCarousel([]);
  }
}

async function loadRewardRules(id) {
  try {
    const res = await fetch(
      `${SERVER_API}/api/bigscore/reward_rule/gets-by-id-club/${id}`,
      {
        headers: { "X-CLIENT-REQUEST": "HERO" },
      }
    );
    const data = await res.json();
    if (data.code === 200) {
      allRules = data.data;
      const curMode = document.getElementById("gameModeSelect").value;
      renderRule(curMode);
    } else {
      // Toastify({ text: "Không tìm thấy dữ liệu CLB", backgroundColor: "orange" }).showToast();
    }
  } catch (e) {
    console.error(e);
  }
}

function bindEditButtons() {
    const editBtns = document.querySelectorAll(".btn-edit-customer");
    editBtns.forEach(btn => {
        btn.addEventListener("click", function() {
            const form = document.getElementById("addCustomerForm");
            const modalTitle = document.querySelector("#addCustomerModal .modal-title");
            
            form.dataset.mode = "edit";
            form.dataset.id = this.dataset.id;
            modalTitle.innerText = "Sửa Thông Tin";

            form.querySelector("#amount_received").value = this.dataset.amount_received;
            form.querySelector("#name_customer").value = this.dataset.name_customer;

            let dateVal = this.dataset.received_at;
            if(dateVal && dateVal.includes(" ")) dateVal = dateVal.split(" ")[0]; // lấy phần ngày
            form.querySelector("#received_at").value = dateVal;

            const imgPreview = document.getElementById("previewAddImage");
            const placeholder = document.getElementById("placeholderText");
            if(this.dataset.image_url && !this.dataset.image_url.includes("placeholder")) {
                imgPreview.src = this.dataset.image_url;
                imgPreview.style.display = "block";
                placeholder.style.display = "none";
            }

            form.querySelector("#amount_received").dispatchEvent(new Event('input'));

            // Show Modal
            const modal = new bootstrap.Modal(document.getElementById('addCustomerModal'));
            modal.show();
        });
    });
}

document.addEventListener("DOMContentLoaded", function () {
  const numberFields = ["default_currency", "daily_increment", "total_amount", "amount_received"];

  numberFields.forEach(id => {
      setupInput(id, 9 , "số tiền");
  });

  const moneyField = ["amount_received"];
  moneyField.forEach(id => {
      setupInput(id, 9, "tiền thưởng");
  });

  const nameField = ["name_customer"];
  nameField.forEach(id => {
      setupInput(id, 25, "ký tự");
  });

//thể lệ

  const quill = new Quill("#editor-container", {
    theme: "snow",
    modules: { toolbar: "#toolbar" },
  });

  quill.on('text-change', () => {
  if (quill.getLength() > 2000) {
    console.log("HTML:", quill.root.innerHTML);
console.log("TEXT:", quill.getText());

    quill.deleteText(2000, quill.getLength())
  }
})

// mô tả giải thưởng

  const quill_reward = new Quill("#editor-container-reward", {
    theme: "snow",
    modules: { toolbar: "#toolbarReward" },
  });

    quill_reward.on('text-change', () => {
  if (quill_reward.getLength() > 2000) {
    quill_reward.deleteText(2000, quill.getLength())
  }
})

  const savedRewardRuleId = localStorage.getItem("club_id");
  if (savedRewardRuleId) {
    document.getElementById("loginForm").classList.add("d-none");
    document.getElementById("mainContent").classList.remove("d-none");
    const defaultMode = 0; 
    loadRewardRules(savedRewardRuleId);
    loadWinners(savedRewardRuleId, defaultMode);
    
    const hiddenClubId = document.getElementById("club_id");
    if(hiddenClubId) hiddenClubId.value = savedRewardRuleId;
  }

  document.getElementById("clubIdForm").addEventListener("submit", async (e) => {
    e.preventDefault();
    const club_id = document.getElementById("clubIdInput").value.trim();
    if (club_id) {
        await loadRewardRules(club_id);
        if (allRules && allRules.length > 0) {
            localStorage.setItem("club_id", club_id);
            document.getElementById("loginForm").classList.add("d-none");
            document.getElementById("mainContent").classList.remove("d-none");
            loadWinners(club_id, 0);
            document.getElementById("club_id").value = club_id; 
        } else {
            Toastify({ text: "Không tìm thấy CLB này!", backgroundColor: "red" }).showToast();
        }
    }
  });

  document.getElementById("gameModeSelect").addEventListener("change", (e) => {
    const selectedMode = e.target.value;
    const clubId = localStorage.getItem("club_id");
    document.querySelector(".title_game_mode").textContent = e.target.options[e.target.selectedIndex].text;

    renderRule(selectedMode);
    loadWinners(clubId, selectedMode);
  });

  const imageInput = document.getElementById("image");
  if (imageInput) {
    imageInput.addEventListener("change", function (e) {
      const file = e.target.files[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = function (ev) {
          const img = document.getElementById("previewAddImage");
          img.src = ev.target.result;
          img.style.display = "block";
          document.getElementById("placeholderText").style.display = "none";
        };
        reader.readAsDataURL(file);
      }
    });
  }


  document.getElementById("ruleRewardForm").addEventListener("submit", async function (e) {
      e.preventDefault();
      const payload = {
        club_id: localStorage.getItem("club_id"),
        game_mode: Number(document.getElementById("gameMode").value),
        default_currency: Number(document.getElementById("default_currency").value),
        daily_increment: Number(document.getElementById("daily_increment").value),
        total_amount: Number(document.getElementById("total_amount").value),
        status: document.getElementById("selectStatus").value === "true",
        rule_details: quill.root.innerHTML,
        reward_content: quill_reward.root.innerHTML,
      };

      const url = mode === "edit"
          ? `${SERVER_API}/api/bigscore/reward_rule/update`
          : `${SERVER_API}/api/bigscore/reward_rule/create`;
      
      if (mode === "edit") payload.id = idRewardRule;

      try {
        const res = await fetch(url, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            "X-CLIENT-REQUEST": "HERO",
          },
          body: JSON.stringify(payload),
        });
        const data = await res.json();
        if (data.code === 200) {
          Toastify({ text: "Lưu thành công!", backgroundColor: "green" }).showToast();
          bootstrap.Modal.getInstance(document.getElementById("ruleRewardModal")).hide();
          loadRewardRules(localStorage.getItem("club_id"));
        } else {
          Toastify({ text: data.message || "Lỗi lưu dữ liệu", backgroundColor: "red" }).showToast();
        }
      } catch (e) {
        console.error(e);
      }
    });


  document.getElementById("addCustomerForm").addEventListener("submit", async (e) => {
      e.preventDefault();
      const formEl = e.target;
      const formData = new FormData(formEl);
      const currentMode = formEl.dataset.mode || "create"; // create | edit
      
      // Get Values
      const jsonData = {
        club_id: localStorage.getItem("club_id"),
        name_customer: formData.get("name_customer"),
        game_mode: Number(document.getElementById("gameModeSelect").value),
        amount_received: formData.get("amount_received"),
        received_at: formData.get("received_at"),
      };

      let url = `${SERVER_API}/api/bigscore/customer/create`;
      
      if (currentMode === "edit") {
          jsonData.id = formEl.dataset.id;
          url = `${SERVER_API}/api/bigscore/customer/update`;
      }

      // Tạo FormData mới để gửi
      const newFormData = new FormData();
      const imageFile = formData.get("image");
      if (imageFile && imageFile.name) {
        newFormData.append("image", imageFile);
      }
      newFormData.append("jsonData", JSON.stringify(jsonData));


      const submitBtn = document.getElementById("saveCustomerBtn");
      submitBtn.disabled = true;
      submitBtn.innerText = "Đang xử lý...";

      try {
        const res = await fetch(url, {
          method: "POST",
          body: newFormData,
          headers: { Accept: "application/json", "X-CLIENT-REQUEST": "HERO" },
        });
        const result = await res.json();

        if (result.code === 200) {
          Toastify({ text: "Thành công!", backgroundColor: "green" }).showToast();
          

          bootstrap.Modal.getInstance(document.getElementById("addCustomerModal")).hide();
          formEl.reset();
          document.getElementById("previewAddImage").style.display = "none";
          document.getElementById("placeholderText").style.display = "block";

          const mode = document.getElementById("gameModeSelect").value;
          loadWinners(localStorage.getItem("club_id"), mode);
        } else {
          Toastify({ text: "Thất bại: " + result.message, backgroundColor: "red" }).showToast();
        }
      } catch (err) {
        console.error(err);
        Toastify({ text: "Lỗi kết nối server", backgroundColor: "red" }).showToast();
      } finally {
        submitBtn.disabled = false;
        submitBtn.innerText = "Xác nhận trao thưởng";
      }
    });

  document.getElementById("confirmDeleteBtn").addEventListener("click", async function () {
      if (!currentDeleteId) return;
      try {
        const res = await fetch(
          `${SERVER_API}/api/bigscore/customer/delete/${currentDeleteId}`,
          {
            method: "DELETE",
            headers: { "X-CLIENT-REQUEST": "HERO" },
          }
        );
        const data = await res.json();
        if (data.code === 200) {
          Toastify({ text: "Đã xóa thành công", backgroundColor: "green" }).showToast();
          bootstrap.Modal.getInstance(document.getElementById("deleteCustomerModal")).hide();
          // Reload List
          const mode = document.getElementById("gameModeSelect").value;
          loadWinners(localStorage.getItem("club_id"), mode);
        } else {
          Toastify({ text: "Lỗi khi xóa", backgroundColor: "red" }).showToast();
        }
      } catch (e) {
        console.error(e);
      }
    });
    

    document.getElementById("confirmDeleteReward").addEventListener("click", async () => {
        if (!idRewardRule) return;
        try {
            const res = await fetch(`${SERVER_API}/api/bigscore/reward_rule/delete/${idRewardRule}`, {
                method: "DELETE",
                headers: { Accept: "application/json", "X-CLIENT-REQUEST": "HERO" },
            });
            const data = await res.json();
            if (data.code === 200) {
                Toastify({ text: "Đã xóa cấu hình", backgroundColor: "green" }).showToast();
                bootstrap.Modal.getInstance(document.getElementById("deleteRewardRuleModal")).hide();
                
                // Clear Editor
                quill.clipboard.dangerouslyPasteHTML("");
                quill_reward.clipboard.dangerouslyPasteHTML("");
                loadRewardRules(localStorage.getItem("club_id"));
            } else {
                Toastify({ text: "Xóa thất bại", backgroundColor: "red" }).showToast();
            }
        } catch (e) { console.error(e); }
    });


    const addCustomerModal = document.getElementById('addCustomerModal');
    addCustomerModal.addEventListener('hidden.bs.modal', function () {
        const form = document.getElementById('addCustomerForm');
        form.reset();
        form.dataset.mode = 'create'; // Reset mode về create
        document.querySelector("#addCustomerModal .modal-title").innerText = "Trao Thưởng";
        document.getElementById("previewAddImage").style.display = "none";
        document.getElementById("placeholderText").style.display = "block";
        document.getElementById("amount_in_words").innerText = "";
    });

    const ruleRewardModal = document.getElementById("ruleRewardModal");
    ruleRewardModal.addEventListener("show.bs.modal", (event) => {
        const button = event.relatedTarget;
        mode = button.getAttribute("data-mode");
        const title = ruleRewardModal.querySelector(".modal-title");
        const gameModeSelectVal = document.getElementById("gameModeSelect").value;
        
        if (mode === "add") {
            title.textContent = "Tạo Giải Thưởng Mới";
            document.getElementById("ruleRewardForm").reset();
            document.getElementById("gameMode").value = gameModeSelectVal;
            quill.clipboard.dangerouslyPasteHTML("");
            quill_reward.clipboard.dangerouslyPasteHTML("");
        } else {
            title.textContent = "Chỉnh Sửa Cấu Hình";
            const rule = allRules.find((r) => r.game_mode == gameModeSelectVal);
            if (rule) {
                document.getElementById("gameMode").value = rule.game_mode;
                document.getElementById("default_currency").value = rule.default_currency;
                document.getElementById("daily_increment").value = rule.daily_increment;
                document.getElementById("total_amount").value = rule.total_amount;
                document.getElementById("selectStatus").value = String(rule.status);
                quill.clipboard.dangerouslyPasteHTML(rule.rule_details || "");
                quill_reward.clipboard.dangerouslyPasteHTML(rule.reward_content || "");
                const fields = ["default_currency", "daily_increment", "total_amount","amount_received"];
                fields.forEach(id => {
                    const input = document.getElementById(id);
                    if (input) {
                        input.value = rule[id] || 0; 
                        input.dispatchEvent(new Event('input')); 
                    }
                });
            }
        }
    });

});